"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.testCalculations = void 0;
var _chai = require("chai");
var _pickers = require("test/utils/pickers");
var _describeGregorianAdapter = require("./describeGregorianAdapter.utils");
/**
 * To check if the date has the right offset even after changing it's date parts,
 * we convert it to a different timezone that always has the same offset,
 * then we check that both dates have the same hour value.
 */ // We change to
const expectSameTimeInMonacoTZ = (adapter, value) => {
  const valueInMonacoTz = adapter.setTimezone(value, 'Europe/Monaco');
  (0, _chai.expect)(adapter.getHours(value)).to.equal(adapter.getHours(valueInMonacoTz));
};
const testCalculations = ({
  adapter,
  adapterTZ,
  adapterFr,
  setDefaultTimezone,
  getLocaleFromDate
}) => {
  const testDateIso = adapter.date(_describeGregorianAdapter.TEST_DATE_ISO_STRING);
  const testDateLastNonDSTDay = adapterTZ.isTimezoneCompatible ? adapterTZ.dateWithTimezone('2022-03-27', 'Europe/Paris') : adapterTZ.date('2022-03-27');
  const testDateLocale = adapter.date(_describeGregorianAdapter.TEST_DATE_LOCALE_STRING);
  describe('Method: date', () => {
    it('should parse strings', () => {
      (0, _chai.expect)(adapter.date(_describeGregorianAdapter.TEST_DATE_ISO_STRING)).toEqualDateTime(testDateIso);
      (0, _chai.expect)(adapter.date(_describeGregorianAdapter.TEST_DATE_LOCALE_STRING)).toEqualDateTime(testDateLocale);
    });
    it('should parse native Date object', () => {
      (0, _chai.expect)(adapter.date(new Date(_describeGregorianAdapter.TEST_DATE_ISO_STRING))).toEqualDateTime(testDateIso);
      (0, _chai.expect)(adapter.date(new Date(_describeGregorianAdapter.TEST_DATE_LOCALE_STRING))).toEqualDateTime(testDateLocale);
    });
    it('should parse already-parsed object', () => {
      (0, _chai.expect)(adapter.date(testDateIso)).toEqualDateTime(testDateIso);
      (0, _chai.expect)(adapter.date(testDateLocale)).toEqualDateTime(testDateLocale);
    });
    it('should parse null', () => {
      (0, _chai.expect)(adapter.date(null)).to.equal(null);
    });
    it('should parse undefined', () => {
      (0, _chai.expect)(Math.abs(adapter.toJsDate(adapter.date(undefined)).getTime() - Date.now())).to.be.lessThan(5);
    });
  });
  describe('Method: dateWithTimezone', () => {
    it('should parse ISO strings', () => {
      if (adapter.isTimezoneCompatible) {
        const test = (timezone, expectedTimezones = timezone) => {
          [adapterTZ, adapterFr].forEach(instance => {
            const dateWithZone = instance.dateWithTimezone(_describeGregorianAdapter.TEST_DATE_ISO_STRING, timezone);
            (0, _chai.expect)(instance.getTimezone(dateWithZone)).to.equal(expectedTimezones);

            // Should keep the time of the value in the UTC timezone
            (0, _chai.expect)(dateWithZone).toEqualDateTime(_describeGregorianAdapter.TEST_DATE_ISO_STRING);
          });
        };
        test('UTC');
        test('system');
        test('America/New_York');
        test('Europe/Paris');
        setDefaultTimezone('America/New_York');
        test('default', 'America/New_York');
        setDefaultTimezone('Europe/Paris');
        test('default', 'Europe/Paris');

        // Reset to the default timezone
        setDefaultTimezone(undefined);
      } else {
        (0, _chai.expect)(adapter.dateWithTimezone(_describeGregorianAdapter.TEST_DATE_ISO_STRING, 'system')).toEqualDateTime(_describeGregorianAdapter.TEST_DATE_ISO_STRING);
        (0, _chai.expect)(adapter.dateWithTimezone(_describeGregorianAdapter.TEST_DATE_ISO_STRING, 'default')).toEqualDateTime(_describeGregorianAdapter.TEST_DATE_ISO_STRING);
      }
    });
    it('should parse locale strings', () => {
      if (adapter.isTimezoneCompatible) {
        const test = timezone => {
          [adapterTZ, adapterFr].forEach(instance => {
            const dateWithZone = instance.dateWithTimezone(_describeGregorianAdapter.TEST_DATE_LOCALE_STRING, timezone);
            (0, _chai.expect)(instance.getTimezone(dateWithZone)).to.equal(timezone);

            // Should keep the time of the date in the target timezone
            (0, _chai.expect)(instance.format(dateWithZone, 'fullTime24h')).to.equal('00:00');
          });
        };
        test('UTC');
        test('system');
        test('America/New_York');
        test('Europe/Paris');
      } else {
        (0, _chai.expect)(adapter.dateWithTimezone(_describeGregorianAdapter.TEST_DATE_LOCALE_STRING, 'system')).toEqualDateTime(_describeGregorianAdapter.TEST_DATE_LOCALE_STRING);
      }
    });
    it('should parse null', () => {
      (0, _chai.expect)(adapter.dateWithTimezone(null, 'system')).to.equal(null);
      if (adapter.isTimezoneCompatible) {
        (0, _chai.expect)(adapter.dateWithTimezone(null, 'UTC')).to.equal(null);
        (0, _chai.expect)(adapter.dateWithTimezone(null, 'America/New_York')).to.equal(null);
      }
    });
    it('should parse undefined', () => {
      if (adapter.isTimezoneCompatible) {
        const testTodayZone = timezone => {
          const dateWithZone = adapterTZ.dateWithTimezone(undefined, timezone);
          (0, _chai.expect)(adapterTZ.getTimezone(dateWithZone)).to.equal(timezone);
          (0, _chai.expect)(adapterTZ.getDiff(dateWithZone, adapterTZ.date(new Date()))).to.be.lessThan(5);
          (0, _chai.expect)(Math.abs(adapterTZ.toJsDate(dateWithZone).getTime() - Date.now())).to.be.lessThan(5);
        };
        testTodayZone('system');
        testTodayZone('UTC');
        testTodayZone('America/New_York');
      } else {
        (0, _chai.expect)(Math.abs(adapterTZ.toJsDate(adapter.dateWithTimezone(undefined, 'system')).getTime() - Date.now())).to.be.lessThan(5);
      }
    });
  });
  it('Method: getTimezone', () => {
    if (!adapter.isTimezoneCompatible) {
      return;
    }
    const testTimezone = (timezone, expectedTimezone = timezone) => {
      (0, _chai.expect)(adapter.getTimezone(adapter.dateWithTimezone(undefined, timezone))).to.equal(expectedTimezone);
    };
    testTimezone('system');
    testTimezone('Europe/Paris');
    testTimezone('America/New_York');
    testTimezone('UTC');
    setDefaultTimezone('America/Chicago');
    testTimezone('default', 'America/Chicago');
    setDefaultTimezone(undefined);
  });
  it('Method: setTimezone', () => {
    if (adapter.isTimezoneCompatible) {
      const test = timezone => {
        setDefaultTimezone(timezone);
        const dateWithLocaleTimezone = adapter.dateWithTimezone(_describeGregorianAdapter.TEST_DATE_ISO_STRING, 'system');
        const dateWithDefaultTimezone = adapter.setTimezone(dateWithLocaleTimezone, 'default');
        (0, _chai.expect)(adapter.getTimezone(dateWithDefaultTimezone)).to.equal(timezone);
      };
      test('America/New_York');
      test('Europe/Paris');

      // Reset to the default timezone
      setDefaultTimezone(undefined);
    } else {
      const systemDate = adapter.dateWithTimezone(_describeGregorianAdapter.TEST_DATE_ISO_STRING, 'system');
      (0, _chai.expect)(adapter.setTimezone(systemDate, 'default')).toEqualDateTime(systemDate);
      (0, _chai.expect)(adapter.setTimezone(systemDate, 'system')).toEqualDateTime(systemDate);
      const defaultDate = adapter.dateWithTimezone(_describeGregorianAdapter.TEST_DATE_ISO_STRING, 'default');
      (0, _chai.expect)(adapter.setTimezone(systemDate, 'default')).toEqualDateTime(defaultDate);
      (0, _chai.expect)(adapter.setTimezone(systemDate, 'system')).toEqualDateTime(defaultDate);
    }
  });
  it('Method: toJsDate', () => {
    (0, _chai.expect)(adapter.toJsDate(testDateIso)).to.be.instanceOf(Date);
    (0, _chai.expect)(adapter.toJsDate(testDateLocale)).to.be.instanceOf(Date);
  });
  it('Method: parseISO', () => {
    (0, _chai.expect)(adapter.parseISO(_describeGregorianAdapter.TEST_DATE_ISO_STRING)).toEqualDateTime(testDateIso);
  });
  it('Method: toISO', () => {
    const outputtedISO = adapter.toISO(testDateIso);
    if (adapter.lib === 'date-fns') {
      // date-fns never suppress useless milliseconds in the end
      (0, _chai.expect)(outputtedISO).to.equal(_describeGregorianAdapter.TEST_DATE_ISO_STRING.replace('.000Z', 'Z'));
    } else {
      (0, _chai.expect)(outputtedISO).to.equal(_describeGregorianAdapter.TEST_DATE_ISO_STRING);
    }
  });
  it('Method: parse', () => {
    // Date time
    (0, _chai.expect)(adapter.parse('10/30/2018 11:44', adapter.formats.keyboardDateTime24h)).toEqualDateTime('2018-10-30T11:44:00.000Z');

    // Date
    (0, _chai.expect)(adapter.parse('10/30/2018', adapter.formats.keyboardDate)).toEqualDateTime('2018-10-30T00:00:00.000Z');

    // Empty string
    (0, _chai.expect)(adapter.parse('', adapter.formats.keyboardDate)).to.equal(null);

    // Invalid input
    (0, _chai.expect)(adapter.isValid(adapter.parse('99/99/9999', adapter.formats.keyboardDate))).to.equal(false);
  });
  it('Method: isNull', () => {
    (0, _chai.expect)(adapter.isNull(null)).to.equal(true);
    (0, _chai.expect)(adapter.isNull(testDateIso)).to.equal(false);
    (0, _chai.expect)(adapter.isNull(testDateLocale)).to.equal(false);
  });
  it('Method: isValid', () => {
    const invalidDate = adapter.date('2018-42-30T11:60:00.000Z');
    (0, _chai.expect)(adapter.isValid(testDateIso)).to.equal(true);
    (0, _chai.expect)(adapter.isValid(testDateLocale)).to.equal(true);
    (0, _chai.expect)(adapter.isValid(invalidDate)).to.equal(false);
    (0, _chai.expect)(adapter.isValid(undefined)).to.equal(true);
    (0, _chai.expect)(adapter.isValid(null)).to.equal(false);
    (0, _chai.expect)(adapter.isValid('2018-42-30T11:60:00.000Z')).to.equal(false);
  });
  describe('Method: getDiff', () => {
    it('should compute the millisecond diff when there is no unit', () => {
      (0, _chai.expect)(adapter.getDiff(testDateIso, adapter.date('2018-10-29T11:44:00.000Z'))).to.equal(86400000);
      (0, _chai.expect)(adapter.getDiff(testDateIso, adapter.date('2018-10-31T11:44:00.000Z'))).to.equal(-86400000);
      (0, _chai.expect)(adapter.getDiff(testDateIso, adapter.date('2018-10-31T11:44:00.000Z'))).to.equal(-86400000);
    });
    it('should compute the diff in the provided unit (ISO)', () => {
      (0, _chai.expect)(adapter.getDiff(testDateIso, adapter.date('2017-09-29T11:44:00.000Z'), 'years')).to.equal(1);
      (0, _chai.expect)(adapter.getDiff(testDateIso, adapter.date('2018-08-29T11:44:00.000Z'), 'months')).to.equal(2);
      (0, _chai.expect)(adapter.getDiff(testDateIso, adapter.date('2018-05-29T11:44:00.000Z'), 'quarters')).to.equal(1);
      (0, _chai.expect)(adapter.getDiff(testDateIso, adapter.date('2018-09-29T11:44:00.000Z'), 'days')).to.equal(31);
      (0, _chai.expect)(adapter.getDiff(testDateIso, adapter.date('2018-09-29T11:44:00.000Z'), 'weeks')).to.equal(4);
      (0, _chai.expect)(adapter.getDiff(testDateIso, adapter.date('2018-09-29T11:44:00.000Z'), 'hours')).to.equal(744);
      (0, _chai.expect)(adapter.getDiff(testDateIso, adapter.date('2018-09-29T11:44:00.000Z'), 'minutes')).to.equal(44640);
      (0, _chai.expect)(adapter.getDiff(testDateIso, adapter.date('2018-10-30T10:44:00.000Z'), 'seconds')).to.equal(3600);
      (0, _chai.expect)(adapter.getDiff(testDateIso, adapter.date('2018-10-30T10:44:00.000Z'), 'milliseconds')).to.equal(3600000);
    });
    it('should compute the diff in the provided unit (locale)', () => {
      (0, _chai.expect)(adapter.getDiff(testDateLocale, adapter.date('2017-09-29'), 'years')).to.equal(1);
      (0, _chai.expect)(adapter.getDiff(testDateLocale, adapter.date('2018-08-29'), 'months')).to.equal(2);
      (0, _chai.expect)(adapter.getDiff(testDateLocale, adapter.date('2018-05-29'), 'quarters')).to.equal(1);
      (0, _chai.expect)(adapter.getDiff(testDateLocale, adapter.date('2018-09-29'), 'days')).to.equal(31);
      (0, _chai.expect)(adapter.getDiff(testDateLocale, adapter.date('2018-09-29'), 'weeks')).to.equal(4);
    });
    it('should compute the diff with string "comparing" param', () => {
      (0, _chai.expect)(adapter.getDiff(testDateLocale, '2017-09-29', 'years')).to.equal(1);
      (0, _chai.expect)(adapter.getDiff(testDateLocale, '2018-08-29', 'months')).to.equal(2);
      (0, _chai.expect)(adapter.getDiff(testDateLocale, '2018-05-29', 'quarters')).to.equal(1);
      (0, _chai.expect)(adapter.getDiff(testDateLocale, '2018-09-29', 'days')).to.equal(31);
      (0, _chai.expect)(adapter.getDiff(testDateLocale, '2018-09-29', 'weeks')).to.equal(4);
    });
  });
  describe('Method: isEqual', () => {
    it('should work in the same timezone', () => {
      (0, _chai.expect)(adapter.isEqual(adapter.date(null), null)).to.equal(true);
      (0, _chai.expect)(adapter.isEqual(testDateIso, adapter.date(_describeGregorianAdapter.TEST_DATE_ISO_STRING))).to.equal(true);
      (0, _chai.expect)(adapter.isEqual(null, testDateIso)).to.equal(false);
      (0, _chai.expect)(adapter.isEqual(testDateLocale, adapter.date(_describeGregorianAdapter.TEST_DATE_LOCALE_STRING))).to.equal(true);
      (0, _chai.expect)(adapter.isEqual(null, testDateLocale)).to.equal(false);
    });
    it('should work with different timezones', function test() {
      if (!adapter.isTimezoneCompatible) {
        this.skip();
      }
      const dateInLondonTZ = adapterTZ.setTimezone(testDateIso, 'Europe/London');
      const dateInParisTZ = adapterTZ.setTimezone(testDateIso, 'Europe/Paris');
      (0, _chai.expect)(adapterTZ.isEqual(dateInLondonTZ, dateInParisTZ)).to.equal(true);
    });
  });
  describe('Method: isSameYear', () => {
    it('should work in the same timezone', () => {
      (0, _chai.expect)(adapter.isSameYear(testDateIso, adapter.date('2018-10-01T00:00:00.000Z'))).to.equal(true);
      (0, _chai.expect)(adapter.isSameYear(testDateIso, adapter.date('2019-10-01T00:00:00.000Z'))).to.equal(false);
      (0, _chai.expect)(adapter.isSameYear(testDateLocale, adapter.date('2018-10-01T00:00:00.000Z'))).to.equal(true);
      (0, _chai.expect)(adapter.isSameYear(testDateLocale, adapter.date('2019-10-01T00:00:00.000Z'))).to.equal(false);
    });
    it('should work with different timezones', function test() {
      if (!adapter.isTimezoneCompatible) {
        this.skip();
      }

      // Both dates below have the same timestamp, but they are not in the same year when represented in their respective timezone.
      // The adapter should still consider that they are in the same year.
      const dateInLondonTZ = adapterTZ.endOfYear(adapterTZ.setTimezone(testDateIso, 'Europe/London'));
      const dateInParisTZ = adapterTZ.setTimezone(dateInLondonTZ, 'Europe/Paris');
      (0, _chai.expect)(adapterTZ.isSameYear(dateInLondonTZ, dateInParisTZ)).to.equal(true);
      (0, _chai.expect)(adapterTZ.isSameYear(dateInParisTZ, dateInLondonTZ)).to.equal(true);
    });
  });
  describe('Method: isSameMonth', () => {
    it('should work in the same timezone', () => {
      (0, _chai.expect)(adapter.isSameMonth(testDateIso, adapter.date('2018-10-01T00:00:00.000Z'))).to.equal(true);
      (0, _chai.expect)(adapter.isSameMonth(testDateIso, adapter.date('2019-10-01T00:00:00.000Z'))).to.equal(false);
      (0, _chai.expect)(adapter.isSameMonth(testDateLocale, adapter.date('2018-10-01T00:00:00.000Z'))).to.equal(true);
      (0, _chai.expect)(adapter.isSameMonth(testDateLocale, adapter.date('2019-10-01T00:00:00.000Z'))).to.equal(false);
    });
    it('should work with different timezones', function test() {
      if (!adapter.isTimezoneCompatible) {
        this.skip();
      }

      // Both dates below have the same timestamp, but they are not in the same month when represented in their respective timezone.
      // The adapter should still consider that they are in the same month.
      const dateInLondonTZ = adapterTZ.endOfMonth(adapterTZ.setTimezone(testDateIso, 'Europe/London'));
      const dateInParisTZ = adapterTZ.setTimezone(dateInLondonTZ, 'Europe/Paris');
      (0, _chai.expect)(adapterTZ.isSameMonth(dateInLondonTZ, dateInParisTZ)).to.equal(true);
      (0, _chai.expect)(adapterTZ.isSameMonth(dateInParisTZ, dateInLondonTZ)).to.equal(true);
    });
  });
  describe('Method: isSameDay', () => {
    it('should work in the same timezone', () => {
      (0, _chai.expect)(adapter.isSameDay(testDateIso, adapter.date('2018-10-30T00:00:00.000Z'))).to.equal(true);
      (0, _chai.expect)(adapter.isSameDay(testDateIso, adapter.date('2019-10-30T00:00:00.000Z'))).to.equal(false);
      (0, _chai.expect)(adapter.isSameDay(testDateLocale, adapter.date('2018-10-30T00:00:00.000Z'))).to.equal(true);
      (0, _chai.expect)(adapter.isSameDay(testDateLocale, adapter.date('2019-10-30T00:00:00.000Z'))).to.equal(false);
    });
    it('should work with different timezones', function test() {
      if (!adapter.isTimezoneCompatible) {
        this.skip();
      }

      // Both dates below have the same timestamp, but they are not in the same day when represented in their respective timezone.
      // The adapter should still consider that they are in the same day.
      const dateInLondonTZ = adapterTZ.endOfDay(adapterTZ.setTimezone(testDateIso, 'Europe/London'));
      const dateInParisTZ = adapterTZ.setTimezone(dateInLondonTZ, 'Europe/Paris');
      (0, _chai.expect)(adapterTZ.isSameDay(dateInLondonTZ, dateInParisTZ)).to.equal(true);
      (0, _chai.expect)(adapterTZ.isSameDay(dateInParisTZ, dateInLondonTZ)).to.equal(true);
    });
  });
  describe('Method: isSameHour', () => {
    it('should work in the same timezone', () => {
      (0, _chai.expect)(adapter.isSameHour(testDateIso, adapter.date('2018-10-30T11:00:00.000Z'))).to.equal(true);
      (0, _chai.expect)(adapter.isSameHour(testDateIso, adapter.date('2018-10-30T12:00:00.000Z'))).to.equal(false);
    });
    it('should work with different timezones', function test() {
      if (!adapter.isTimezoneCompatible) {
        this.skip();
      }

      // Both dates below have the same timestamp, but they are not in the same day when represented in their respective timezone.
      // The adapter should still consider that they are in the same day.
      const dateInLondonTZ = adapterTZ.setTimezone(testDateIso, 'Europe/London');
      const dateInParisTZ = adapterTZ.setTimezone(dateInLondonTZ, 'Europe/Paris');
      (0, _chai.expect)(adapterTZ.isSameHour(dateInLondonTZ, dateInParisTZ)).to.equal(true);
      (0, _chai.expect)(adapterTZ.isSameHour(dateInParisTZ, dateInLondonTZ)).to.equal(true);
    });
  });
  describe('Method: isAfter', () => {
    it('should work with the same timezone', () => {
      (0, _chai.expect)(adapter.isAfter(adapter.date(), testDateIso)).to.equal(true);
      (0, _chai.expect)(adapter.isAfter(testDateIso, adapter.date())).to.equal(false);
      (0, _chai.expect)(adapter.isAfter(adapter.date(), testDateLocale)).to.equal(true);
      (0, _chai.expect)(adapter.isAfter(testDateLocale, adapter.date())).to.equal(false);
    });
    it('should work with different timezones', function test() {
      if (!adapter.isTimezoneCompatible) {
        this.skip();
      }
      const dateInLondonTZ = adapterTZ.endOfDay(adapterTZ.setTimezone(testDateIso, 'Europe/London'));
      const dateInParisTZ = adapterTZ.addMinutes(adapterTZ.endOfDay(adapterTZ.setTimezone(testDateIso, 'Europe/Paris')), 30);
      (0, _chai.expect)(adapter.isAfter(dateInLondonTZ, dateInParisTZ)).to.equal(true);
      (0, _chai.expect)(adapter.isAfter(dateInParisTZ, dateInLondonTZ)).to.equal(false);
    });
  });
  describe('Method: isAfterYear', () => {
    it('should work in the same timezone', () => {
      const nextYearIso = adapter.addYears(testDateIso, 1);
      (0, _chai.expect)(adapter.isAfterYear(nextYearIso, testDateIso)).to.equal(true);
      (0, _chai.expect)(adapter.isAfterYear(testDateIso, nextYearIso)).to.equal(false);
      const nextYearLocale = adapter.addYears(testDateLocale, 1);
      (0, _chai.expect)(adapter.isAfterYear(nextYearLocale, testDateLocale)).to.equal(true);
      (0, _chai.expect)(adapter.isAfterYear(testDateLocale, nextYearLocale)).to.equal(false);
    });
    it('should work with different timezones', function test() {
      if (!adapter.isTimezoneCompatible) {
        this.skip();
      }

      // Both dates below have the same timestamp, but they are not in the same year when represented in their respective timezone.
      // The adapter should still consider that they are in the same year.
      const dateInLondonTZ = adapterTZ.endOfYear(adapterTZ.setTimezone(testDateIso, 'Europe/London'));
      const dateInParisTZ = adapterTZ.setTimezone(dateInLondonTZ, 'Europe/Paris');
      (0, _chai.expect)(adapterTZ.isAfterYear(dateInLondonTZ, dateInParisTZ)).to.equal(false);
      (0, _chai.expect)(adapterTZ.isAfterYear(dateInParisTZ, dateInLondonTZ)).to.equal(false);
    });
  });
  describe('Method: isAfterDay', () => {
    it('should work with the same timezone', () => {
      const nextDayIso = adapter.addDays(testDateIso, 1);
      (0, _chai.expect)(adapter.isAfterDay(nextDayIso, testDateIso)).to.equal(true);
      (0, _chai.expect)(adapter.isAfterDay(testDateIso, nextDayIso)).to.equal(false);
      const nextDayLocale = adapter.addDays(testDateLocale, 1);
      (0, _chai.expect)(adapter.isAfterDay(nextDayLocale, testDateLocale)).to.equal(true);
      (0, _chai.expect)(adapter.isAfterDay(testDateLocale, nextDayLocale)).to.equal(false);
    });
    it('should work with different timezones', function test() {
      if (!adapter.isTimezoneCompatible) {
        this.skip();
      }

      // Both dates below have the same timestamp, but they are not in the same day when represented in their respective timezone.
      // The adapter should still consider that they are in the same day.
      const dateInLondonTZ = adapterTZ.endOfDay(adapterTZ.setTimezone(testDateIso, 'Europe/London'));
      const dateInParisTZ = adapterTZ.setTimezone(dateInLondonTZ, 'Europe/Paris');
      (0, _chai.expect)(adapterTZ.isAfterDay(dateInLondonTZ, dateInParisTZ)).to.equal(false);
      (0, _chai.expect)(adapterTZ.isAfterDay(dateInParisTZ, dateInLondonTZ)).to.equal(false);

      // Both dates below have the same day when represented in their respective timezone,
      // But not when represented in the same timezone
      // The adapter should consider that they are not in the same day
      const dateInLondonTZ2 = adapterTZ.startOfDay(adapterTZ.setTimezone(testDateIso, 'Europe/London'));
      const dateInParisTZ2 = adapterTZ.addHours(adapterTZ.setTimezone(dateInLondonTZ2, 'Europe/Paris'), -1);
      (0, _chai.expect)(adapterTZ.isAfterDay(dateInLondonTZ2, dateInParisTZ2)).to.equal(true);
      (0, _chai.expect)(adapterTZ.isAfterDay(dateInParisTZ2, dateInLondonTZ2)).to.equal(false);
    });
  });
  describe('Method: isBefore', () => {
    it('should work with the same timezone', () => {
      (0, _chai.expect)(adapter.isBefore(testDateIso, adapter.date())).to.equal(true);
      (0, _chai.expect)(adapter.isBefore(adapter.date(), testDateIso)).to.equal(false);
      (0, _chai.expect)(adapter.isBefore(testDateLocale, adapter.date())).to.equal(true);
      (0, _chai.expect)(adapter.isBefore(adapter.date(), testDateLocale)).to.equal(false);
    });
    it('should work with different timezones', function test() {
      if (!adapter.isTimezoneCompatible) {
        this.skip();
      }
      const dateInLondonTZ = adapterTZ.endOfDay(adapterTZ.setTimezone(testDateIso, 'Europe/London'));
      const dateInParisTZ = adapterTZ.addMinutes(adapterTZ.endOfDay(adapterTZ.setTimezone(testDateIso, 'Europe/Paris')), 30);
      (0, _chai.expect)(adapter.isBefore(dateInLondonTZ, dateInParisTZ)).to.equal(false);
      (0, _chai.expect)(adapter.isBefore(dateInParisTZ, dateInLondonTZ)).to.equal(true);
    });
  });
  describe('Method: isBeforeYear', () => {
    it('should work in the same timezone', () => {
      const nextYearIso = adapter.addYears(testDateIso, -1);
      (0, _chai.expect)(adapter.isBeforeYear(nextYearIso, testDateIso)).to.equal(true);
      (0, _chai.expect)(adapter.isBeforeYear(testDateIso, nextYearIso)).to.equal(false);
      const nextYearLocale = adapter.addYears(testDateLocale, -1);
      (0, _chai.expect)(adapter.isBeforeYear(nextYearLocale, testDateLocale)).to.equal(true);
      (0, _chai.expect)(adapter.isBeforeYear(testDateLocale, nextYearLocale)).to.equal(false);
    });
    it('should work with different timezones', function test() {
      if (!adapter.isTimezoneCompatible) {
        this.skip();
      }

      // Both dates below have the same timestamp, but they are not in the same year when represented in their respective timezone.
      // The adapter should still consider that they are in the same year.
      const dateInLondonTZ = adapterTZ.endOfYear(adapterTZ.setTimezone(testDateIso, 'Europe/London'));
      const dateInParisTZ = adapterTZ.setTimezone(dateInLondonTZ, 'Europe/Paris');
      (0, _chai.expect)(adapterTZ.isBeforeYear(dateInLondonTZ, dateInParisTZ)).to.equal(false);
      (0, _chai.expect)(adapterTZ.isBeforeYear(dateInParisTZ, dateInLondonTZ)).to.equal(false);
    });
  });
  describe('Method: isBeforeDay', () => {
    it('should work with the same timezone', () => {
      const previousDayIso = adapter.addDays(testDateIso, -1);
      (0, _chai.expect)(adapter.isBeforeDay(previousDayIso, testDateIso)).to.equal(true);
      (0, _chai.expect)(adapter.isBeforeDay(testDateIso, previousDayIso)).to.equal(false);
      const previousDayLocale = adapter.addDays(testDateLocale, -1);
      (0, _chai.expect)(adapter.isBeforeDay(previousDayLocale, testDateLocale)).to.equal(true);
      (0, _chai.expect)(adapter.isBeforeDay(testDateLocale, previousDayLocale)).to.equal(false);
    });
    it('should work with different timezones', function test() {
      if (!adapter.isTimezoneCompatible) {
        this.skip();
      }

      // Both dates below have the same timestamp, but they are not in the same day when represented in their respective timezone.
      // The adapter should still consider that they are in the same day.
      const dateInLondonTZ = adapterTZ.endOfDay(adapterTZ.setTimezone(testDateIso, 'Europe/London'));
      const dateInParisTZ = adapterTZ.setTimezone(dateInLondonTZ, 'Europe/Paris');
      (0, _chai.expect)(adapterTZ.isBeforeDay(dateInLondonTZ, dateInParisTZ)).to.equal(false);
      (0, _chai.expect)(adapterTZ.isBeforeDay(dateInParisTZ, dateInLondonTZ)).to.equal(false);

      // Both dates below have the same day when represented in their respective timezone,
      // But not when represented in the same timezone
      // The adapter should consider that they are not in the same day
      const dateInLondonTZ2 = adapterTZ.endOfDay(adapterTZ.setTimezone(testDateIso, 'Europe/London'));
      const dateInParisTZ2 = adapterTZ.addHours(adapterTZ.setTimezone(dateInLondonTZ2, 'Europe/Paris'), -1);
      (0, _chai.expect)(adapterTZ.isBeforeDay(dateInLondonTZ2, dateInParisTZ2)).to.equal(false);
      (0, _chai.expect)(adapterTZ.isBeforeDay(dateInParisTZ2, dateInLondonTZ2)).to.equal(true);
    });
  });
  describe('Method: isWithinRange', () => {
    it('should work on simple examples', () => {
      (0, _chai.expect)(adapter.isWithinRange(adapter.date('2019-10-01T00:00:00.000Z'), [adapter.date('2019-09-01T00:00:00.000Z'), adapter.date('2019-11-01T00:00:00.000Z')])).to.equal(true);
      (0, _chai.expect)(adapter.isWithinRange(adapter.date('2019-12-01T00:00:00.000Z'), [adapter.date('2019-09-01T00:00:00.000Z'), adapter.date('2019-11-01T00:00:00.000Z')])).to.equal(false);
      (0, _chai.expect)(adapter.isWithinRange(adapter.date('2019-10-01'), [adapter.date('2019-09-01'), adapter.date('2019-11-01')])).to.equal(true);
      (0, _chai.expect)(adapter.isWithinRange(adapter.date('2019-12-01'), [adapter.date('2019-09-01'), adapter.date('2019-11-01')])).to.equal(false);
    });
    it('should use inclusivity of range', () => {
      (0, _chai.expect)(adapter.isWithinRange(adapter.date('2019-09-01T00:00:00.000Z'), [adapter.date('2019-09-01T00:00:00.000Z'), adapter.date('2019-12-01T00:00:00.000Z')])).to.equal(true);
      (0, _chai.expect)(adapter.isWithinRange(adapter.date('2019-12-01T00:00:00.000Z'), [adapter.date('2019-09-01T00:00:00.000Z'), adapter.date('2019-12-01T00:00:00.000Z')])).to.equal(true);
      (0, _chai.expect)(adapter.isWithinRange(adapter.date('2019-09-01'), [adapter.date('2019-09-01'), adapter.date('2019-12-01')])).to.equal(true);
      (0, _chai.expect)(adapter.isWithinRange(adapter.date('2019-12-01'), [adapter.date('2019-09-01'), adapter.date('2019-12-01')])).to.equal(true);
    });
  });
  it('Method: startOfYear', () => {
    const expected = '2018-01-01T00:00:00.000Z';
    (0, _chai.expect)(adapter.startOfYear(testDateIso)).toEqualDateTime(expected);
    (0, _chai.expect)(adapter.startOfYear(testDateLocale)).toEqualDateTime(expected);
  });
  it('Method: startOfMonth', () => {
    const expected = '2018-10-01T00:00:00.000Z';
    (0, _chai.expect)(adapter.startOfMonth(testDateIso)).toEqualDateTime(expected);
    (0, _chai.expect)(adapter.startOfMonth(testDateLocale)).toEqualDateTime(expected);
  });
  it('Method: startOfWeek', () => {
    const expected = adapter.lib === 'luxon' ? '2018-10-29T00:00:00.000Z' : '2018-10-28T00:00:00.000Z';
    (0, _chai.expect)(adapter.startOfWeek(testDateIso)).toEqualDateTime(expected);
    (0, _chai.expect)(adapter.startOfWeek(testDateLocale)).toEqualDateTime(expected);
  });
  it('Method: startOfDay', () => {
    const expected = '2018-10-30T00:00:00.000Z';
    (0, _chai.expect)(adapter.startOfDay(testDateIso)).toEqualDateTime(expected);
    (0, _chai.expect)(adapter.startOfDay(testDateLocale)).toEqualDateTime(expected);
  });
  it('Method: endOfYear', () => {
    const expected = '2018-12-31T23:59:59.999Z';
    (0, _chai.expect)(adapter.endOfYear(testDateIso)).toEqualDateTime(expected);
    (0, _chai.expect)(adapter.endOfYear(testDateLocale)).toEqualDateTime(expected);
  });
  describe('Method: endOfMonth', () => {
    it('should handle basic usecases', () => {
      const expected = '2018-10-31T23:59:59.999Z';
      (0, _chai.expect)(adapter.endOfMonth(testDateIso)).toEqualDateTime(expected);
      (0, _chai.expect)(adapter.endOfMonth(testDateLocale)).toEqualDateTime(expected);
    });
    it('should update the offset when entering DST', function test() {
      if (!adapterTZ.isTimezoneCompatible) {
        this.skip();
      }
      expectSameTimeInMonacoTZ(adapterTZ, testDateLastNonDSTDay);
      expectSameTimeInMonacoTZ(adapterTZ, adapterTZ.endOfMonth(testDateLastNonDSTDay));
    });
  });
  it('Method: endOfWeek', () => {
    const expected = adapter.lib === 'luxon' ? '2018-11-04T23:59:59.999Z' : '2018-11-03T23:59:59.999Z';
    (0, _chai.expect)(adapter.endOfWeek(testDateIso)).toEqualDateTime(expected);
    (0, _chai.expect)(adapter.endOfWeek(testDateLocale)).toEqualDateTime(expected);
  });
  it('Method: endOfDay', () => {
    const expected = '2018-10-30T23:59:59.999Z';
    (0, _chai.expect)(adapter.endOfDay(testDateIso)).toEqualDateTime(expected);
    (0, _chai.expect)(adapter.endOfDay(testDateLocale)).toEqualDateTime(expected);
  });
  it('Method: addYears', () => {
    (0, _chai.expect)(adapter.addYears(testDateIso, 2)).toEqualDateTime('2020-10-30T11:44:00.000Z');
    (0, _chai.expect)(adapter.addYears(testDateIso, -2)).toEqualDateTime('2016-10-30T11:44:00.000Z');
  });
  describe('Method: addMonths', () => {
    it('should handle basic usecases', () => {
      (0, _chai.expect)(adapter.addMonths(testDateIso, 2)).toEqualDateTime('2018-12-30T11:44:00.000Z');
      (0, _chai.expect)(adapter.addMonths(testDateIso, -2)).toEqualDateTime('2018-08-30T11:44:00.000Z');
      (0, _chai.expect)(adapter.addMonths(testDateIso, 3)).toEqualDateTime('2019-01-30T11:44:00.000Z');
    });
    it('should update the offset when entering DST', function test() {
      if (!adapterTZ.isTimezoneCompatible) {
        this.skip();
      }
      expectSameTimeInMonacoTZ(adapterTZ, testDateLastNonDSTDay);
      expectSameTimeInMonacoTZ(adapterTZ, adapterTZ.addMonths(testDateLastNonDSTDay, 1));
    });
  });
  describe('Method: addWeeks', () => {
    it('should handle basic usecases', () => {
      (0, _chai.expect)(adapter.addWeeks(testDateIso, 2)).toEqualDateTime('2018-11-13T11:44:00.000Z');
      (0, _chai.expect)(adapter.addWeeks(testDateIso, -2)).toEqualDateTime('2018-10-16T11:44:00.000Z');
    });
    it('should update the offset when entering DST', function test() {
      if (!adapterTZ.isTimezoneCompatible) {
        this.skip();
      }
      expectSameTimeInMonacoTZ(adapterTZ, testDateLastNonDSTDay);
      expectSameTimeInMonacoTZ(adapterTZ, adapterTZ.addWeeks(testDateLastNonDSTDay, 1));
    });
  });
  describe('Method: addWeeks', () => {
    it('should handle basic usecases', () => {
      (0, _chai.expect)(adapter.addDays(testDateIso, 2)).toEqualDateTime('2018-11-01T11:44:00.000Z');
      (0, _chai.expect)(adapter.addDays(testDateIso, -2)).toEqualDateTime('2018-10-28T11:44:00.000Z');
    });
    it('should update the offset when entering DST', function test() {
      if (!adapterTZ.isTimezoneCompatible) {
        this.skip();
      }
      expectSameTimeInMonacoTZ(adapterTZ, testDateLastNonDSTDay);
      expectSameTimeInMonacoTZ(adapterTZ, adapterTZ.addDays(testDateLastNonDSTDay, 1));
    });
  });
  it('Method: addHours', () => {
    (0, _chai.expect)(adapter.addHours(testDateIso, 2)).toEqualDateTime('2018-10-30T13:44:00.000Z');
    (0, _chai.expect)(adapter.addHours(testDateIso, -2)).toEqualDateTime('2018-10-30T09:44:00.000Z');
    (0, _chai.expect)(adapter.addHours(testDateIso, 15)).toEqualDateTime('2018-10-31T02:44:00.000Z');
  });
  it('Method: addMinutes', () => {
    (0, _chai.expect)(adapter.addMinutes(testDateIso, 2)).toEqualDateTime('2018-10-30T11:46:00.000Z');
    (0, _chai.expect)(adapter.addMinutes(testDateIso, -2)).toEqualDateTime('2018-10-30T11:42:00.000Z');
    (0, _chai.expect)(adapter.addMinutes(testDateIso, 20)).toEqualDateTime('2018-10-30T12:04:00.000Z');
  });
  it('Method: addSeconds', () => {
    (0, _chai.expect)(adapter.addSeconds(testDateIso, 2)).toEqualDateTime('2018-10-30T11:44:02.000Z');
    (0, _chai.expect)(adapter.addSeconds(testDateIso, -2)).toEqualDateTime('2018-10-30T11:43:58.000Z');
    (0, _chai.expect)(adapter.addSeconds(testDateIso, 70)).toEqualDateTime('2018-10-30T11:45:10.000Z');
  });
  it('Method: getYear', () => {
    (0, _chai.expect)(adapter.getYear(testDateIso)).to.equal(2018);
  });
  it('Method: getMonth', () => {
    (0, _chai.expect)(adapter.getMonth(testDateIso)).to.equal(9);
  });
  it('Method: getDate', () => {
    (0, _chai.expect)(adapter.getDate(testDateIso)).to.equal(30);
  });
  it('Method: getHours', () => {
    (0, _chai.expect)(adapter.getHours(testDateIso)).to.equal(11);
  });
  it('Method: getMinutes', () => {
    (0, _chai.expect)(adapter.getMinutes(testDateIso)).to.equal(44);
  });
  it('Method: getSeconds', () => {
    (0, _chai.expect)(adapter.getSeconds(testDateIso)).to.equal(0);
  });
  it('Method: getMilliseconds', () => {
    (0, _chai.expect)(adapter.getMilliseconds(testDateIso)).to.equal(0);
  });
  it('Method: setYear', () => {
    (0, _chai.expect)(adapter.setYear(testDateIso, 2011)).toEqualDateTime('2011-10-30T11:44:00.000Z');
  });
  it('Method: setMonth', () => {
    (0, _chai.expect)(adapter.setMonth(testDateIso, 4)).toEqualDateTime('2018-05-30T11:44:00.000Z');
  });
  it('Method: setDate', () => {
    (0, _chai.expect)(adapter.setDate(testDateIso, 15)).toEqualDateTime('2018-10-15T11:44:00.000Z');
  });
  it('Method: setHours', () => {
    (0, _chai.expect)(adapter.setHours(testDateIso, 0)).toEqualDateTime('2018-10-30T00:44:00.000Z');
  });
  it('Method: setMinutes', () => {
    (0, _chai.expect)(adapter.setMinutes(testDateIso, 12)).toEqualDateTime('2018-10-30T11:12:00.000Z');
  });
  it('Method: setSeconds', () => {
    (0, _chai.expect)(adapter.setSeconds(testDateIso, 11)).toEqualDateTime('2018-10-30T11:44:11.000Z');
  });
  it('Method: setMilliseconds', () => {
    (0, _chai.expect)(adapter.setMilliseconds(testDateIso, 11)).toEqualDateTime('2018-10-30T11:44:00.011Z');
  });
  it('Method: getDaysInMonth', () => {
    (0, _chai.expect)(adapter.getDaysInMonth(testDateIso)).to.equal(31);
    (0, _chai.expect)(adapter.getDaysInMonth(testDateLocale)).to.equal(31);
    (0, _chai.expect)(adapter.getDaysInMonth(adapter.addMonths(testDateIso, 1))).to.equal(30);
  });
  it('Method: getNextMonth', () => {
    (0, _chai.expect)(adapter.getNextMonth(testDateIso)).toEqualDateTime('2018-11-30T11:44:00.000Z');
  });
  it('Method: getPreviousMonth', () => {
    (0, _chai.expect)(adapter.getPreviousMonth(testDateIso)).toEqualDateTime('2018-09-30T11:44:00.000Z');
  });
  it('Method: getMonthArray', () => {
    const monthArray = adapter.getMonthArray(testDateIso);
    let expectedDate = adapter.date('2018-01-01T00:00:00.000Z');
    monthArray.forEach(month => {
      (0, _chai.expect)(month).toEqualDateTime(expectedDate);
      expectedDate = adapter.addMonths(expectedDate, 1);
    });
  });
  it('Method: mergeDateAndTime', () => {
    const mergedDate = adapter.mergeDateAndTime(testDateIso, adapter.date('2018-01-01T14:15:16.000Z'));
    (0, _chai.expect)(adapter.toJsDate(mergedDate)).toEqualDateTime('2018-10-30T14:15:16.000Z');
  });
  it('Method: getWeekdays', () => {
    const weekDays = adapter.getWeekdays();
    let date = adapter.startOfWeek(testDateIso);
    weekDays.forEach(dayLabel => {
      (0, _chai.expect)(adapter.format(date, 'weekday').startsWith(dayLabel)).to.equal(true);
      date = adapter.addDays(date, 1);
    });
  });
  describe('Method: getWeekArray', () => {
    it('should work without timezones', () => {
      const weekArray = adapter.getWeekArray(testDateIso);
      let expectedDate = adapter.startOfWeek(adapter.startOfMonth(testDateIso));
      (0, _chai.expect)(weekArray).to.have.length(5);
      weekArray.forEach(week => {
        (0, _chai.expect)(week).to.have.length(7);
        week.forEach(day => {
          (0, _chai.expect)(day).toEqualDateTime(expectedDate);
          expectedDate = adapter.addDays(expectedDate, 1);
        });
      });
    });
    it('should respect the DST', function test() {
      if (!adapterTZ.isTimezoneCompatible) {
        this.skip();
      }
      const referenceDate = adapterTZ.dateWithTimezone('2022-03-17', 'Europe/Paris');
      const weekArray = adapterTZ.getWeekArray(referenceDate);
      let expectedDate = adapter.startOfWeek(adapter.startOfMonth(referenceDate));
      (0, _chai.expect)(weekArray).to.have.length(5);
      weekArray.forEach(week => {
        (0, _chai.expect)(week).to.have.length(7);
        week.forEach(day => {
          (0, _chai.expect)(adapterTZ.startOfDay(day)).toEqualDateTime(adapterTZ.startOfDay(expectedDate));
          expectedDate = adapterTZ.addDays(expectedDate, 1);
          (0, _chai.expect)((0, _pickers.getDateOffset)(adapterTZ, day)).to.equal(adapterTZ.isAfter(day, testDateLastNonDSTDay) ? 120 : 60);
        });
      });
    });
    it('should respect the locale of the adapter, not the locale of the date', function test() {
      const dateFr = adapterFr.dateWithTimezone('2022-03-17', 'default');
      const weekArray = adapter.getWeekArray(dateFr);
      if (getLocaleFromDate) {
        (0, _chai.expect)(getLocaleFromDate(weekArray[0][0])).to.match(/en/);
      }

      // Week should start on Monday (28th of March) for adapters supporting locale-based week start.
      (0, _chai.expect)(adapter.getDate(weekArray[0][0])).to.equal(adapter.lib === 'luxon' ? 28 : 27);
    });
  });
  it('Method: getWeekNumber', () => {
    (0, _chai.expect)(adapter.getWeekNumber(testDateIso)).to.equal(44);
  });
  it('Method: getYearRange', () => {
    const yearRange = adapter.getYearRange(testDateIso, adapter.setYear(testDateIso, 2124));
    (0, _chai.expect)(yearRange).to.have.length(107);
    (0, _chai.expect)(adapter.getYear(yearRange[yearRange.length - 1])).to.equal(2124);
    const emptyYearRange = adapter.getYearRange(testDateIso, adapter.setYear(testDateIso, adapter.getYear(testDateIso) - 1));
    (0, _chai.expect)(emptyYearRange).to.have.length(0);
  });
};
exports.testCalculations = testCalculations;