from dataclasses import dataclass, field
import json
from typing import Dict, List, Optional
from datetime import date
import importlib.util
import os
import sys


@dataclass
class ExtensionTag:
    text: str
    addedOn: date

@dataclass
class ExtensionRating:
    rating: int
    addedOn: date
    bumped: Optional[bool]

@dataclass
class ExtensionReview:
    markdown: str
    addedOn: date

@dataclass
class ExtensionArtist:
    id: str  # channelId
    name: str

@dataclass
class ExtensionSong:
    id: str  # videoId
    name: str
    tags: List[ExtensionTag] = field(default_factory=list)
    reviews: List[ExtensionReview] = field(default_factory=list)

@dataclass
class ExtensionAlbum:
    id: str  # playlistId
    name: str
    release_year: int
    thumbnail_url: str
    tags: List[ExtensionTag] = field(default_factory=list)
    artists: List[str] = field(default_factory=list)  # List of artist IDs
    songIds: List[str] = field(default_factory=list)
    ratings: List[ExtensionRating] = field(default_factory=list)
    reviews: List[ExtensionReview] = field(default_factory=list)

@dataclass
class ExtensionExportedData:
    songs: Dict[str, ExtensionSong]
    default_albums: Dict[str, ExtensionAlbum]
    uploaded_albums: Dict[str, ExtensionAlbum]
    all_albums: Dict[str, ExtensionAlbum]
    artists: Dict[str, ExtensionArtist]
    song_tags: List[ExtensionTag]
    album_tags: List[ExtensionTag]
    song_to_album: Dict[str, str]


def import_from_file(module_name, file_path):
    spec = importlib.util.spec_from_file_location(module_name, file_path)
    if spec is None:
        raise ModuleNotFoundError(f"Could not find module or invalid path: {file_path}")
    module = importlib.util.module_from_spec(spec)
    sys.modules[module_name] = module
    loader = spec.loader
    if loader is None:
        raise ModuleNotFoundError(f"Loader not found for: {file_path}")
    loader.exec_module(module)
    return module


module_name = "extension_common"
module_path = os.path.join("/Users/srajan/code/ytmusic++/apis", module_name + ".py")
extension_common = import_from_file(module_name, module_path)


music_data = extension_common.ExtensionExportedData.from_export("./input/music_export.json")

annotation_data = json.load(open("./input/annotation_export.json"))